/**
* Cells_filter.java
* Created on 25 October 2020 by Manuel Guillermo Forero-Vargas
* e-mail: mgforero@yahoo.es
*
* Function: This plug-in seeks to eliminate the false cells found, comparing the
* difference in between the minimum and maximum level of grey in each cell. If 
* the value is below a threshold, the cell is eliminated. Finally all the cells
* are relabelled.
* 
* Input: 8 and 16 bit images. The labelled image is modified.
* Output: The labelled image
*
* Copyright (c) 2020 by Manuel Guillermo Forero-Vargas
* e-mail: mgforero@yahoo.es
*
* This plugin is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License version 2
* as published by the Free Software Foundation.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this plugin; if not, write to the Free Software
* Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
import ij.*;
import ij.process.*;
import ij.gui.*;
import ij.plugin.*;

public class Cells_filter implements PlugIn
{
    @Override
    public void run(String arg)
    {
        boolean flLabel,relabel=false;
        boolean fNumber=true;
        int a,i,j,minDif=100,minmodifiedlabel,type,z;
        short[] pixelsLabel;
        //----------------------------------------------------------------------
        //Gets the image to be processed and the mask image
        int[] wList=ij.WindowManager.getIDList();
        if (wList==null||wList.length<2)
        {
            ij.IJ.showMessage("Cell filter","This plugin requires two iimages.");
            return;
        }
        String[] titles=new String[wList.length];
        for (i=0;i<wList.length;i++)
        {
            ij.ImagePlus imp=ij.WindowManager.getImage(wList[i]);
            titles[i]=imp!=null?imp.getTitle():"";
        }
        GenericDialog gd=new GenericDialog("Cell filter");
        gd.addChoice("Original_image:",titles,titles[0]);
        gd.addChoice("Labelled_image:",titles,titles[1]);
        //----------------------------------------------------------------------
        /*Creates a dialog box, allowing the user to enter the required
        minimum and maximum volume.*/
        gd.addNumericField("Minimum grey level diference",minDif,0);
        gd.addCheckbox("Show number of remaining objects",fNumber);
        gd.showDialog();
        if (gd.wasCanceled())
            return;
        int index1=gd.getNextChoiceIndex();
        ij.ImagePlus impOri=ij.WindowManager.getImage(wList[index1]);//Original image
        int index2=gd.getNextChoiceIndex();
        ij.ImagePlus impLabel=ij.WindowManager.getImage(wList[index2]);//Labelled image
        minDif=(int)gd.getNextNumber();
        type=impOri.getType();
       /* if(type!=impLabel.getType())
        {
            IJ.error("Both images must be of the same type.");
            return;
        }*/
        //----------------------------------------------------------------------
        //Charge the original image
        int width=impOri.getWidth();
        int height=impOri.getHeight();
        ImageStack stackOri=impOri.getStack();
        ImageStack stackLabel=impLabel.getStack();
        //----------------------------------------------------------------------
        //Get number of labelled regions in the imjage
        int numRegions=0;
        for (z=0;z<stackLabel.getSize();z++)
        {
            pixelsLabel=(short[])((ShortProcessor)stackLabel.getProcessor(z+1)).getPixels();
            for(i=0;i<height*width;i++)
                if (pixelsLabel[i]>numRegions)
                    numRegions=pixelsLabel[i];
        }
        numRegions++;
        //----------------------------------------------------------------------
        int[]	labels=new int[numRegions];//Table of labels
        double[]    entropyLabels=new double[numRegions];//Entropy of each labeled region
        double[][] histNorm=new double[numRegions][256];//Histogram of each labeled regionn
        int[]	minLabels=new int[numRegions];//Minimum of each labeled region
        int[]	maxLabels=new int[numRegions];//Maximum of each labeled region
        int[]	sizeLabels=new int[numRegions];//Size of each labeled region

        minmodifiedlabel=numRegions;
        //Gets the minimum and maximum grey level of each labelled region
        //Initialises table of of min labels
        for(i=1;i<numRegions;i++)
        {
            minLabels[i]=255;
        }
        //----------------------------------------------------------------------
        //Get minimum and maximum grey level, size and histogram of each labelled region
        for (z=0;z<stackLabel.getSize();z++)
        {
            byte[]  pixelsOri=(byte[])((ByteProcessor)stackOri.getProcessor(z+1)).getPixels();
            pixelsLabel=(short[])((ShortProcessor)stackLabel.getProcessor(z+1)).getPixels();
            for(i=0;i<height*width;i++)
                if (pixelsLabel[i]!=0)
                {
                    a=pixelsOri[i]&0xff;
                    if(minLabels[pixelsLabel[i]]>a)
                        minLabels[pixelsLabel[i]]=a;
                    if(maxLabels[pixelsLabel[i]]<a)
                        maxLabels[pixelsLabel[i]]=a;
                    histNorm[pixelsLabel[i]][a]++;
                    sizeLabels[pixelsLabel[i]]++;
                }
        }
        //----------------------------------------------------------------------
        //Initialises table of labels and normalise histogram of each region
        double entropy;
        for(i=1;i<numRegions;i++)
        {
            labels[i]=i;
            for (a=1;a<256;a++)
            {
                histNorm[i][a]/=sizeLabels[i];
                if(histNorm[i][a]!=0)
                    entropyLabels[i]-=histNorm[i][a]*Math.log(histNorm[i][a])/Math.log(2);
            }
         //   IJ.log("label="+i+" entro="+entropyLabels[i]);
        }
        //----------------------------------------------------------------------
        //Verifies if there is at least one object smaller or bigger than the thresholds and eliminates them
        for (i=1;i<numRegions;i++)
            if (maxLabels[i]-minLabels[i]<minDif||entropyLabels[i]<7.09)
            {
                labels[i]=0;
                if (relabel==false)
                {
                    relabel=true;
                    minmodifiedlabel=i;//Identifies the smallest modified label
                }
            }
        //Minimises the region indexes in the list
        if (relabel)
        {
            for(i=minmodifiedlabel;i<numRegions;i++)
            {
                flLabel=true;
                for(j=i;j<numRegions;j++)//If a label is equal to its position on the list...
                    if(labels[j]==i)
                    {
                        flLabel=false;
                        break;
                    }
                while(flLabel)
                {
                    flLabel=false;
                    for(j=i;j<numRegions;j++)
                        if(labels[j]>i)
                        {
                            labels[j]--;
                            flLabel=true;
                        }
                    for(j=i;j<numRegions;j++)
                        if(labels[j]==i)
                        {
                            flLabel=false;
                            break;
                        }
                }
            }
            //Labels the regions in the resulting stack again
            for (z=0;z<stackLabel.getSize();z++)
            {
                pixelsLabel=(short[])((ShortProcessor)stackLabel.getProcessor(z+1)).getPixels();
                for(i=0;i<height*width;i++)
                    pixelsLabel[i]=(short)labels[pixelsLabel[i]];
            }
        }
//------------------------------------------------------------------------------
//Counts the remaining objects
        if (fNumber)
        {
            if (relabel)
            {
                int max=0;
                for (i=1;i<numRegions;i++)
                    if (max<labels[i])
                        max=labels[i];
                IJ.log("Number of cells: "+max);
            }
            else
                IJ.log("Number of cells: "+--numRegions);
        }
        impLabel.updateAndDraw();
    }

    void showAbout()
    {
        IJ.showMessage("About VolumeFilter_...",
        "This plug-in filter eliminates the 3D-objects that are smaller or bigger than a predefined volume.");
    }
}